#!/usr/bin/python
# -*- coding: utf-8 -*-
'''
Created on 10.11.2017
    
    Indigo_main class providing a user interface.  
    Copyright (C) 2018  Esa Pursiheimo

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <https://www.gnu.org/licenses/>.
    
@author: Esa Pursiheimo
'''

import sys
import os
from PySide.QtGui import *
from PySide.QtCore import *
from ind_gui import Ui_MainWindow
import Indigo_test
from Indigo_test import Gis_app
import csv
import pickle

import matplotlib
matplotlib.use('Qt4Agg')
matplotlib.rcParams['backend.qt4']='PySide'
import matplotlib.gridspec as gridspec
import numpy as np
import matplotlib.pyplot as plt
from matplotlib.figure import Figure
from matplotlib.backends.backend_qt4agg import FigureCanvasQTAgg as FigureCanvas

from matplotlib.backends.backend_qt4agg import NavigationToolbar2QT as NavToolBar

from case_gui import Ui_Form
from PySide import QtGui
from shutil import copy2
from ast import literal_eval
from Classes.System import System
import itertools
from textwrap import wrap
from distutils.dir_util import copy_tree

from _collections import defaultdict

# Main program class
class INDIGO_main(QMainWindow,Ui_MainWindow):
    def __init__(self):
        super(INDIGO_main,self).__init__()
        self.setupUi(self)
        
        # Gis widget intialised
        self.gis = None
        # Gis widget status (0 = inactive, 1 = active)
        self.gis_status = 0
        
        # Figure and canvas definition for each figure widget
        # Result graph
        self.fig1 = Figure(figsize=(11.3,3.4), dpi=100, facecolor=(1,1,1))
        self.canvas1 = FigureCanvas(self.fig1)
        self.canvas1.setParent(self.graph_widget)
        self.toolbar = NavigationToolbar(self.canvas1,self)
        self.toolbar.setParent(self.nav_widget)
        # Monthly graph
        self.fig2 = Figure(figsize=(4.8,3.1), dpi=100, facecolor=(1,1,1))
        self.canvas2 = FigureCanvas(self.fig2)
        self.canvas2.setParent(self.month_widget)
        # Sensitivity analysis graph
        self.fig3 = Figure(figsize=(7.3,3.5), dpi=100, facecolor=(1,1,1))
        self.canvas3 = FigureCanvas(self.fig3)
        self.canvas3.setParent(self.sens_widget)
        #Economic analysis canvas
        self.fig4 = Figure(figsize=(5.1,3.8), dpi=100, facecolor=(1,1,1))
        self.canvas4 = FigureCanvas(self.fig4)
        self.canvas4.setParent(self.profit_plot)
        #Emission analysis canvas
        self.fig5 = Figure(figsize=(5.2,4.5), dpi=100, facecolor=(1,1,1))
        self.canvas5 = FigureCanvas(self.fig5)
        self.canvas5.setParent(self.emi_plot)
        
        
        # Modify canvas style        
        self.canvas1.setStyleSheet("background-color: rgb(255,0,0); margin:5px; border:1px solid rgb(0, 255, 0); ")
        self.canvas2.setStyleSheet("background-color: rgb(255,0,0); margin:5px; border:1px solid rgb(0, 255, 0); ")
        
        # Result dictionary init
        self.result_prod_dic = defaultdict(dict)
        self.result_sup_dic = defaultdict(dict)
        self.result_sto_dic = defaultdict(list)
        self.result_res_dic = defaultdict(list)
        self.dem = []
        self.month_dic = defaultdict(dict)
        self.value_dic = {}
        self.m_hours = [0,744,1416,2160,2880,3624,4344,5088,5832,6552,7296,8016,8760]
        self.annual_dic = {}
        self.annual_dic['DC production cost'] = 0
        self.annual_dic['DC annual consumption'] = 0
        self.annual_dic['DC total emission'] = 0
        self.sens_dic = {}
        self.sens_unit = {}
        self.ref_dic = {}
        self.unit_type = {}
        self.af_dic = defaultdict(dict)
        self.cap_dic = defaultdict(dict)
        self.current_case = None
        self.current_scen = None
        self.profit_dic = defaultdict(dict)
        self.emi_dic = defaultdict(dict)
        
        # Set location combo box items
        flist = os.listdir('.\\Data')
        for f in flist:
            if f[:9] == 'resources':
                #self.loc_combo.addItem('_'.join(f.split('.')[0].split('_')[1:]))
                vec = f.split('.')[0].split('_')
                if len(vec) == 2:
                    self.loc_combo.addItem(vec[1])
        
        # Get cases from GIS-folder
        #self.dir_list = next(os.walk('.\\GIS'))[1]
        self.dir_list = os.listdir('.\\GIS')
        for i in self.dir_list:
            self.case_combo.addItem(i)
            self.sens_case.addItem(i)
            self.res_case_combo.addItem(i)
        case_name = self.case_combo.currentText()
        path_name = '.\\GIS\\'+case_name+'\\'
        self.update_scenario_list(0)
        self.update_scenario_list(1)
        
        # initialise peiod length
        self.period = 0
        self.case_scen_name = ""
        
        # Model construction - parameter table: unit type selection     
        self.unit_selection.addItem('Consumers')
        self.unit_selection.addItem('Producers')
        self.unit_selection.addItem('Pipeline')
        self.unit_selection.addItem('Supply')
        self.unit_selection.addItem('Storage')
        self.unit_selection.currentIndexChanged.connect(lambda: self.update_data_table())
        self.save_changes.clicked.connect(lambda: self.save_data_table())
        
        # Time series settings (ts = consumer types, res = resources)
        self.ts_indx = 0
        self.ts_dic = defaultdict(list)
        self.res_dic = defaultdict(list)
        self.sel_ts.clicked.connect(lambda: self.browse_ts(self.ts_name))
        self.sel_ts_2.clicked.connect(lambda: self.browse_ts(self.ts_name_2))
        self.insert_ts.clicked.connect(lambda: self.import_ts(self.ts_dic))
        self.insert_ts_2.clicked.connect(lambda: self.import_ts(self.res_dic))
        
        # Define signals from case buttons
        self.load_case.clicked.connect(lambda: self.load_gis_case())
        self.save_case.clicked.connect(lambda: self.save_gis_case())
        self.save_scenario.clicked.connect(lambda: self.save_gis_scenario())
        self.import_case.clicked.connect(lambda: self.import_gis_case())
        self.save_copy.clicked.connect(lambda: self.make_case_copy())
        self.delete_scenario.clicked.connect(lambda: self.delete_scenarios())
        self.upd_reftable.clicked.connect(lambda: self.update_ref_table())
        self.save_refcase.clicked.connect(lambda: self.update_ref_dic())
        # Reference case table exists (0 = No, 1 = Yes)
        self.ref_tab_status = 0
        
        # Select case in analysis tab and update scenario list
        self.case_combo.currentIndexChanged.connect(lambda: self.update_scenario_list(0))
        # Select case in sensitivity analysis tab and update scenario list
        self.sens_case.currentIndexChanged.connect(lambda: self.update_scenario_list(1))
        
        # Sensitivity analysis signals
        self.sens_scen.activated.connect(lambda: self.update_sensitivity_table())
        self.sens_unit_list.activated.connect(lambda: self.update_capacity_value())
        self.run_sens.clicked.connect(lambda: self.run_sensitivity())
        
        # REsult graph week selection
        self.graph_spin.valueChanged.connect(lambda: self.update_result_graph())
        
        # Get width of sensitivity table and hide it
        self.mx = self.sens_table.geometry().width()
        self.sens_table.hide()
        
        # Combobox for country selection activated
        self.loc_combo.currentIndexChanged.connect(lambda: self.set_res_file())
        
        # Load results button
        self.load_results.clicked.connect(lambda: self.load_result_graph())
        
        # Result graph type selection
        self.prod_radio.clicked.connect(lambda: self.check_result_type())
        self.sup_radio.clicked.connect(lambda: self.check_result_type())
        self.sto_radio.clicked.connect(lambda: self.check_result_type())
        self.res_radio.clicked.connect(lambda: self.check_result_type())
        # Producer type is default
        self.graf_status = 1
        
        # Launch scenario results and utilisation factors
        self.scen_results.clicked.connect(lambda: self.month_plot())
        #self.af_results.clicked.connect(lambda: self.utilisation_factors())
        
        # Show duration curves
        self.dur_button.clicked.connect(lambda: self.show_duration_curve())
        
        # TURHAA???
        plt.rcParams['backend'] = "qt4agg"
        plt.rcParams['backend.qt4'] = "PySide"
        
        # Run analysis button      
        self.run_case.clicked.connect(lambda: self.create_and_run_model())
        
        # Button showing Total consumption of system
        self.tot_cons.hide()
        self.tot_cons.clicked.connect(lambda: self.show_total_consumption())
        # Testing reference
        #self.test_ref.clicked.connect(lambda: self.build_and_run_reference())
        
        self.tabWidget_2.currentChanged.connect(lambda: self.update_data_table())
        
        # Show main window
        self.show()
    
    #-------------------------------------------------------------------------------------------------------------------------------------
    
    # Create model system and read data from case folder
    def create_and_run_model(self):
        
        # Case details
        case = self.case_combo.currentText()
        self.current_case = case
        case_path = '.\\GIS\\'+case+'\\'
        try:
            scen = self.case_list.currentItem().text()
        except:
            msgBox = QMessageBox()
            msgBox.setText("Select a scenario")
            msgBox.exec_()
            return 0
        self.current_scen = scen
        self.case_scen_name = case+' - '+scen
        
        
        # Update run log 
        self.run_log.clear()       
        self.run_log.append('START: Run case: '+case+' - '+scen)
        QtGui.QApplication.processEvents()

        # Dictionaries initialised
        par_dic = defaultdict(list)
        ts_dic = defaultdict(list)
        nod_dic = {}
        loc_dic = {}
        
        # read nodes and node locations
        with open(case_path+'dh_nodes_'+scen+'.txt','r') as ll:
            for l in ll:
                vec = l.strip('\n').split('\t')
                loc_dic[vec[0]] = literal_eval(vec[1])
        with open(case_path+'prod_con_'+scen+'.txt','r') as pp:
            for p in pp:
                vec = p.strip('\n').split('\t')
                nod_dic[vec[0]] = vec[1]        
        
        # Get consumption time series from con_ts.pickle in case folder
        cdic = case_path+'con_ts.pickle'
        
        # Check if pickle-file exists
        if os.path.exists(cdic):
            with open(cdic, 'rb') as handle:
                ts_dic = pickle.load(handle)
        else:
            msgBox = QMessageBox()
            msgBox.setText("No consumption timeseries available: con_ts.pickle")
            msgBox.exec_()
            return 0
        
        # Get period length
        self.period = self.period_len.value()
        
        # Read tariffs from System_cost.txt
        with open(case_path+'System_cost_'+scen+'.txt','r') as sp:
            for s in sp:
                vec = s.strip('\n').split(' ')
                if vec[0] == 'Energy':
                    e_tariff = [float(vec[1])]*int(self.period)
                elif vec[0] == 'Capacity':
                    c_tariff = float(vec[1])
                else:
                    location = vec[0]
                # Create system and initialise

        self.system = System()
        self.system.initialise(location)
        
        # Create cooling system (case name, True if network, energy tariff, capacity tariff)
        self.system.addCoolingSystem(case, True, e_tariff, c_tariff)
        self.system.initNetwork(case,scen)
                
        # Read component data from System_data.txt
        
        p_indx = 0
        s_indx = 0
        self.result_prod_dic = {}
        self.result_sup_dic = {}
        self.result_res_dic = {}
        self.unit_type = {}
        self.af_dic.clear()
        self.cap_dic.clear()
        self.profit_dic.clear()
        self.emi_dic.clear()
        with open(case_path+'System_data_'+scen+'.txt','r') as fp:
            for f in fp:
                data_vec = f.strip('\n').split('\t')
                self.run_log.append(' '.join(data_vec))
                QApplication.processEvents()
                if len(data_vec) > 1:
                    if data_vec[0][0] == 'C':
                        coord = loc_dic[nod_dic[data_vec[0]]]
                        c_indx = data_vec[0][1:]
                        c_type = data_vec[1]
                        demand = ts_dic[c_type][:int(self.period)]

                        self.system.addConsumer(case,c_indx,demand)
                    elif data_vec[0][0] == 'L':
                        pass
                    elif data_vec[0][0] == 'P':
                        self.result_prod_dic[data_vec[0]] = {}
                        coord = loc_dic[nod_dic[data_vec[0]]]
                        p_indx = data_vec[0][1:]
                        self.unit_type[data_vec[0]] = data_vec[1]
                        par = []
                        for i in range(2,8):
                            par.append(float(data_vec[i]))
                        self.system.addProducer(case,p_indx,par[0],par[1],par[2],par[3],par[4],par[5])
                        self.cap_dic['Producer'][data_vec[0]] = par[0]
                    elif data_vec[0][0] == 'T':
                        par = []
                        self.unit_type[data_vec[1]] = data_vec[1]
                        for i in range(2,7):
                            par.append(float(data_vec[i]))    
                        # TaHaN TARVITAAN KORJAUS, yksi parametri puuttuu (investointi, nyt lisatty 150) - ei loydy tiedostosta!
                        self.system.defineSimpleStorage(case,data_vec[1],par[0],par[1],par[2],par[3],par[4])
                        self.cap_dic['Storage'][data_vec[1]] = par[0]
                    elif data_vec[0][0] == 'S':
                        self.result_sup_dic[data_vec[0]] = {}
                        self.unit_type[data_vec[0]] = data_vec[1]
                        s_indx = data_vec[0][1:]
                        par = []
                        for i in range(3,8):
                            par.append(float(data_vec[i]))
                        self.system.addSupply(case,s_indx,data_vec[1],data_vec[2],par[0],par[1],par[2],par[3],par[4])
                        self.cap_dic['Supply'][data_vec[0]] = par[2]
                        
        
        # Build and run model    
        self.run_log.append('Start optimisation...')
        QApplication.processEvents()
        feas_status = self.system.runOptimisation(False,case,self.period)
        if feas_status == 0:
            self.run_log.append('END: Model infeasible - check model parameters')
            msgBox = QMessageBox()
            msgBox.setText("Infeasible model - check model parameters")
            msgBox.exec_()
            return 0
        else:
            print('Optimisation ready')
            self.system.runEconomicAnalysis(20)   
            
            self.scen_name.setText(self.case_scen_name)
            if self.ref_check.isChecked():
                self.run_log.append('Running reference case...')
                QApplication.processEvents()
                self.build_and_run_reference(case,scen)
            #print(self.system.getPaybackPeriod())
            #print(self.system.getCumulativeCosts())
            #print(self.system.getCumulativeCostsAndRevenues())
            print(self.system.getNumberOfCycles(case,'dc storage'))
            self.get_result_data(self.system,case,scen,case_path,location)
            self.update_result_graph()
            self.update_month_graph()
            self.update_res_table()
            self.update_af_table()
            self.update_economic_results()
            self.update_emission_results()
            self.run_log.append('END: Optimisation ready')
    
            msgBox = QMessageBox()
            msgBox.setText("Optimal solution found - see results")
            msgBox.exec_()
            return 1
    
    #------------------------------------------------------------------------------------------------------------------------------------
    # Build reference case system and run all building cases
    def build_and_run_reference(self,case,scen):
        # Case details
        #case = self.current_case
        #self.period = self.period_len.value()
        #case = self.case_combo.currentText()
        case_path = '.\\GIS\\'+case+'\\'
        #scen = self.current_scen
        #scen = self.case_list.currentItem().text()
        # Read tariffs from System_cost.txt
        with open(case_path+'System_cost_'+scen+'.txt','r') as sp:
            for s in sp:
                vec = s.strip('\n').split(' ')
                if vec[0] == 'Energy':
                    e_tariff = [float(vec[1])]*self.period
                elif vec[0] == 'Capacity':
                    c_tariff = float(vec[1])
                else:
                    location = vec[0]
        # Get consumption time series from con_ts.pickle in case folder
        cdic = case_path+'con_ts.pickle'
        # Check if pickle-file exists
        if os.path.exists(cdic):
            with open(cdic, 'rb') as handle:
                ts_dic = pickle.load(handle)
        else:
            msgBox = QMessageBox()
            msgBox.setText("No consumption timeseries available: con_ts.pickle")
            msgBox.exec_()
            return 0
        # Open reference_data_*scenario*.txt file and start reading lines
        self.ref_system = []
        # Create system and initialise
        system = System()
        system.initialise(location)
        with open(case_path+'reference_data_'+scen+'.txt','r') as fp:
            for f in fp:
                data_vec = f.strip('\n').split('\t')
                if data_vec[0][0] == 'C':
                    # Create cooling system (building name, True if network, energy tariff, capacity tariff)
                    c_sys = data_vec[0]
                    system.addCoolingSystem(c_sys, False, e_tariff, c_tariff)
                    # Get index number and demand for consumer
                    c_indx = data_vec[0][1:]
                    c_type = data_vec[1]
                    demand = ts_dic[c_type][:int(self.period)]
                    # Coordinates obsolete, used (0,0)
                    system.addConsumer(c_sys,c_indx,demand)
                elif data_vec[0][0] == 'P':
                    p_indx = data_vec[0][1:]
                    par = []
                    for i in range(2,8):
                        par.append(float(data_vec[i]))
                    system.addProducer(c_sys,p_indx,par[0],par[1],par[2],par[3],par[4],par[5])   
                elif data_vec[0][0] == 'S':
                    s_indx = data_vec[0][1:]
                    par = []
                    for i in range(3,8):
                        par.append(float(data_vec[i]))
                    system.addSupply(c_sys,s_indx,data_vec[1],data_vec[2],par[0],par[1],par[2],par[3],par[4])    
                else:
                    pass
        # Optimise system with all coolingsystems and run economic analysis
        system.runOptimisation(True,'',self.period)   
        system.runEconomicAnalysis(20)        

        # Get economic data into profit_dic
        self.profit_dic['Reference']['PBP'] = system.getPaybackPeriod()
        self.profit_dic['Reference']['CCR'] = system.getCumulativeCostsAndRevenues()
        self.value_dic['Reference'] = system.getResultsSummary('',location)
                 
    #------------------------------------------------------------------------------------------------------------------------------------
    # Get result data from oemof-system output into dictionaries
    def get_result_data(self,system,case,scen,cpath,location):
        # producer dic
        rdic = self.result_prod_dic
        # supply dic
        sdic = self.result_sup_dic
        # storage dic
        tdic = self.result_sto_dic
        
        # get result data for production
        for c in sorted(rdic.keys()):
            idn = c[1:]
            rdic[c]['District cooling'] = system.getProducerOutput(case,idn)
            self.af_dic['Producer'][c] = system.getUtilisationRate(case,False,idn)

        # get result data for supply
        for s in sorted(sdic.keys()):
            idn = s[1:]
            sdic[s]['Heat'] = system.getSupplyOutput(case,idn,'Heat')
            sdic[s]['Electricity'] = system.getSupplyOutput(case,idn,'Electricity')
            self.af_dic['Supply'][s] = system.getUtilisationRate(case,True,idn)
            
        # get result data for storage 
        tecs = []
        with open('.\\Data\\storages.txt','r') as fp:
            for f in fp:
                tecs.append(f.strip('\n')) 
        for t in tecs:
            vec = system.getStorageState(case,t,True)
            if vec == 0:
                tdic[t] = [0]*self.period
                self.af_dic['Storage'][t] = 0
            else:
                tdic[t] = vec
                self.af_dic['Storage'][t] = system.getNumberOfCycles(case,t)
        
        # get result data for resources 
        self.result_res_dic = system.getUsedResources(case)

        # create common dic for time series pickle-file         
        dic = {}
        dic['Producer'] = rdic
        dic['Supply'] = sdic        
        self.dem = system.getTotalDemand(case)[:self.period]
        dic['Demand'] = self.dem
        dic['Storage'] = tdic
        dic['Resources'] = self.result_res_dic
        with open(cpath+'scen_results_'+scen+'.pickle', 'wb') as handle:
            pickle.dump(dic, handle, protocol=pickle.HIGHEST_PROTOCOL)
        
        # monthly data calculation for production and supply
        pr_mat = {}
        sh_mat = {}
        # Cooling
        for p in sorted(self.result_prod_dic.keys()):
            vec = []
            for i,m in enumerate(self.m_hours):
                if i > 0:
                    vec.append(sum(rdic[p]['District cooling'][self.m_hours[i-1]:m]))
            pr_mat[p] = vec
        # Supply
        for s in sorted(self.result_sup_dic.keys()):
            vec = []    

            for i,m in enumerate(self.m_hours):
                if i > 0:
                    vec.append(sum(sdic[s]['Heat'][self.m_hours[i-1]:m]))                
            sh_mat[s] = vec
        
        # Economic data into profit_dic dictionary
        self.profit_dic['Case']['PBP'] = system.getPaybackPeriod()
        self.profit_dic['Case']['CCR'] = system.getCumulativeCostsAndRevenues()

        if 'Reference' not in self.profit_dic:
            self.profit_dic['Reference']['PBP'] = 20
            self.profit_dic['Reference']['CCR'] = [0]*19

        
        # Emission data into emi_dic 
        self.emi_dic['Operational'] = system.getOperationalEmissions(case,location)
        self.emi_dic['Embedded'] = system.getEmbeddedEmissions(case)
        
        # Put monthly data into month_dic dictionary
        self.month_dic['Producer'] = pr_mat
        self.month_dic['Supply'] = sh_mat
        
        # Save dictionaries into pickle files
        with open(cpath+'scen_months_'+scen+'.pickle', 'wb') as handle:
            pickle.dump(self.month_dic, handle, protocol=pickle.HIGHEST_PROTOCOL)

        # Get result summary into value_dic
        self.value_dic['Case'] = system.getResultsSummary(case,location)
        # Reference data into value_dic if nor reference case is run
        if 'Reference' not in self.value_dic:
            self.value_dic['Reference'] = {'Total cooling (MWh)':'-',
                                           'Total costs (ke)':'-',
                                           'Total emissions (tCO2)':'-',
                                           'Operation emissions (tCO2)':'-',
                                           'Embedded emissions (tCO2)':'-',
                                           'Share of embedded emissions (%)':'-',
                                           'Cooling production costs (e/MWh)':'-',    
                                           'Cooling emissions (kgCO2/MWh)':'-'}
        
        with open(cpath+'scen_values_'+scen+'.pickle', 'wb') as handle:
            pickle.dump(self.value_dic, handle, protocol=pickle.HIGHEST_PROTOCOL)
            
        with open(cpath+'scen_af_'+scen+'.pickle', 'wb') as handle:
            pickle.dump(self.af_dic, handle, protocol=pickle.HIGHEST_PROTOCOL)   
        
        with open(cpath+'scen_cap_'+scen+'.pickle', 'wb') as handle:
            pickle.dump(self.cap_dic, handle, protocol=pickle.HIGHEST_PROTOCOL)  
        
            
        with open(cpath+'scen_types_'+scen+'.pickle', 'wb') as handle:
            pickle.dump(self.unit_type, handle, protocol=pickle.HIGHEST_PROTOCOL)           
            
        with open(cpath+'scen_profit_'+scen+'.pickle', 'wb') as handle:
            pickle.dump(self.profit_dic, handle, protocol=pickle.HIGHEST_PROTOCOL) 

        with open(cpath+'scen_emission_'+scen+'.pickle', 'wb') as handle:
            pickle.dump(self.emi_dic, handle, protocol=pickle.HIGHEST_PROTOCOL)             
                

    #------------------------------------------------------------------------------------------------------------------------------------
    
    # Load results from previously run case-scenarios
    def load_result_file(self,cpath,scen):
        file_name = cpath+'scen_results_'+scen+'.pickle'
        self.case_scen_name = self.res_case_combo.currentText() + ' - '+scen
        
        # open result (producers - supply) pickle file
        if os.path.exists(file_name):
            with open(file_name, 'rb') as handle:
                dic = pickle.load(handle)
        else:
            return -1
        self.result_prod_dic = dic['Producer']
        self.result_sup_dic = dic['Supply']
        self.dem = dic['Demand']
        self.result_sto_dic = dic['Storage']
        self.result_res_dic = dic['Resources']
        
        # load monthly results
        file_name2 = cpath+'scen_months_'+scen+'.pickle'
        if os.path.exists(file_name2):
            with open(file_name2, 'rb') as handle:
                self.month_dic = pickle.load(handle)

        # load indicator results
        file_name3 = cpath+'scen_values_'+scen+'.pickle'
        if os.path.exists(file_name3):
            with open(file_name3, 'rb') as handle:
                self.value_dic = pickle.load(handle)
                
        # load utilisation rate results
        file_name4 = cpath+'scen_af_'+scen+'.pickle'
        if os.path.exists(file_name4):
            with open(file_name4, 'rb') as handle:
                self.af_dic = pickle.load(handle)       
        
        # load utilisation rate results
        file_name4b = cpath+'scen_cap_'+scen+'.pickle'
        if os.path.exists(file_name4b):
            with open(file_name4b, 'rb') as handle:
                self.cap_dic = pickle.load(handle)   
        
        
        # load unit types
        file_name5 = cpath+'scen_types_'+scen+'.pickle'
        if os.path.exists(file_name5):
            with open(file_name5, 'rb') as handle:
                self.unit_type = pickle.load(handle)  

        # load profit results
        file_name6 = cpath+'scen_profit_'+scen+'.pickle'
        if os.path.exists(file_name6):
            with open(file_name6, 'rb') as handle:
                self.profit_dic = pickle.load(handle) 
               
        # load emission results
        file_name7 = cpath+'scen_emission_'+scen+'.pickle'
        if os.path.exists(file_name7):
            with open(file_name7, 'rb') as handle:
                self.emi_dic = pickle.load(handle) 
        
        # Get period length from demand time series
        self.period = len(self.dem)
        # Update case-scenario name
        self.scen_name.setText(self.case_scen_name)
        # Update result, monthly graphs and result table
        self.update_result_graph()
        self.update_month_graph()
        self.update_res_table()
        self.update_af_table()
        self.update_economic_results()
        self.update_emission_results()
        
        
    # open load result window
    def load_result_graph(self):
        case = self.res_case_combo.currentText()
        cpath = '.\\GIS\\'+case+'\\'
        res_wid = scenario_result(None,self,cpath)
        res_wid.show()
    
    # check for producer or supply type in result graph
    def check_result_type(self):
        # is producer radio button checked
        p_sta = self.prod_radio.isChecked()
        s_sta = self.sup_radio.isChecked()
        t_sta = self.sto_radio.isChecked()
        r_sta = self.res_radio.isChecked()
        # Producer
        if p_sta:
            if self.graf_status == 1:
                return
            else:
                self.graf_status = 1
                self.update_result_graph()
        # Supply
        elif s_sta:
            if self.graf_status == 2:
                return
            else:
                self.graf_status = 2
                self.update_result_graph()
        # Storage
        elif t_sta:
            if self.graf_status == 3:
                return
            else:
                self.graf_status = 3
                self.update_result_graph()
        # Resources
        elif r_sta:
            if self.graf_status == 4:
                return
            else:
                self.graf_status = 4
                self.update_result_graph()
    
    # update result graph with 1 week period       
    def update_result_graph(self):
        week = self.graph_spin.value()
        per_len = 168
        if self.period > 0:
            t0 = (week-1)*per_len
            t1 = t0 + per_len
            fig = self.fig1
            canv = self.canvas1
            fig.clf()
            rdic = self.result_prod_dic
            sdic = self.result_sup_dic
            tdic = self.result_sto_dic
            edic = self.result_res_dic
            t = np.arange(per_len)
            # Graph colors for each category
            if self.graf_status == 1:
                #col = ['darkgreen','forestgreen','mediumseagreen','limegreen','lightgreen'] 
                col = ['darkgreen',
                       'lawngreen',
                       'mediumseagreen',
                       'limegreen',
                       'lightgreen']
            elif self.graf_status == 2:
                col = ['indigo',
                       'darkorchid',
                       'mediumorchid',
                       'plum',
                       'pink'] 
            else:
                col = ['navy',
                       'royalblue',
                       'dodgerblue',
                       'skyblue'] 
            res_col = {'HFO': 'dimgrey',
                       'LFO': 'peru',   
                       'NG': 'orangered',   
                       'Chips': 'forestgreen',   
                       'Pellets': 'yellowgreen',   
                       'Sun': 'gold',   
                       'Electricity': 'royalblue',   
                       'Heat': 'salmon',
                       'Cold': 'darkorchid'}
            
            bot = np.zeros(per_len)
            pl = fig.add_subplot(1,1,1)
            # Producer bar graph
            if self.graf_status == 1:
                for i, c in enumerate(sorted(rdic.keys())):
                    y = rdic[c]['District cooling'][t0:t1]
                    pl.bar(t,y,1.0,color=col[i],bottom=bot,align = 'edge',label = c)
                    bot+=y
                # Demand time series
                dmnd = self.dem[t0:t1]
                pl.step(t,dmnd,where='post',linestyle = 'solid', color = 'salmon')
            # Supply bar graph
            elif self.graf_status == 2:
                for i, s in enumerate(sorted(sdic.keys())):
                    y = sdic[s]['Heat'][t0:t1]
                    pl.bar(t,y,1.0,color=col[i],bottom=bot,align = 'edge',label = s)
                    bot+=y
            # Storage state time series
            elif self.graf_status == 3:
                max_y = 0
                for i, s in enumerate(sorted(tdic.keys())):
                    y = tdic[s][t0:t1]
                    max_y = max(max_y,max(tdic[s]))
                    pl.step(t,y,where='post',linestyle='solid',color=col[i],label = s)
                pl.set_ylim(0,1)
            # Resources bar graph
            else:
                for i, e in enumerate(sorted(edic.keys())):
                    y = edic[e].values.tolist()[t0:t1]
                    pl.bar(t,y,1.0,color=res_col[e],bottom=bot,align = 'edge',label = e)
                    bot+=y
            fig.subplots_adjust(top=0.96,right=0.98,left=0.06,bottom=0.07,hspace=0.20)
            pl.legend()
            pl.grid(b=True,color='0.5',linestyle='--')
            pl.set_xlim(0,per_len)
            # Update canvas in gui
            canv.draw()
    
    # update month summary graph in results (producer in left - supply heat production in right)        
    def update_month_graph(self):
        if self.period > 0:
            # Get figure and canvas objects
            fig = self.fig2
            canv = self.canvas2
            # Clear figure
            fig.clf()

            months = ('Jan', 'Feb', 'Mar', 'Apr', 'May','Jun','Jul','Aug','Sep','Oct','Nov','Dec')
            # x-axis
            ind = np.arange(len(months))
            # bottom-vector for bar graph
            bot = np.zeros(len(months))
            # width of bar
            width = 0.45 
            
            pr_mat = self.month_dic['Producer']
            sh_mat = self.month_dic['Supply']
            
            blues = ['SteelBlue','SkyBlue','LightBlue']
            reds = ['IndianRed','Tomato','LightCoral']
            pl = fig.add_subplot(1,1,1)
            # create loop for color vectors
            col1 = itertools.cycle(blues)
            col2 = itertools.cycle(reds)
            # Producer bar
            for i, p in enumerate(sorted(pr_mat.keys())):
                x = ind - width/2
                y = pr_mat[p]
                pl.bar(x,y,width,color=next(col1),bottom=bot,label = p+'_cool')
                bot+=y
            bot = np.zeros(len(months))
            # Heat supply bar
            for i, s in enumerate(sorted(sh_mat.keys())):
                x = ind + width/2
                y = sh_mat[s]
                pl.bar(x,y,width,color=next(col2),bottom=bot,label = s+'_heat')
                bot+=y
            # Graph settings
            pl.set_ylabel('Production (kWh)')
            pl.set_title('Monthly sum')
            pl.set_xticks(ind)
            pl.set_xticklabels(months)
            pl.legend()
            fig.subplots_adjust(top=0.92,right=0.98,left=0.16,bottom=0.07,hspace=0.20)
            for item in ([pl.title, pl.xaxis.label, pl.yaxis.label] + pl.get_xticklabels() + pl.get_yticklabels() + pl.legend().get_texts()):
                item.set_fontsize(8)
                
            canv.draw()
    
    # create month graph collection for scenarios
    def month_plot(self):
        cpath = '.\\GIS\\'+self.res_case_combo.currentText()+'\\'
        
        # Open scenario indicator table window
        self.res_win = scen_values(None,self,cpath)
        self.res_win.show()
        
        # go thru files in case folder to find monthly scenario pickle files
        flist = os.listdir(cpath)
        scen_list = []
        pr_mat = []
        sh_mat = []
        dic = {}
        for f in flist:
            vec = f.split('.')
            if len(vec) > 1:
                if vec[1] == 'pickle':
                    if vec[0][:11] == 'scen_months':
                        scen = vec[0].split('_')[-1]
                        scen_list.append(scen)
        nc = len(scen_list)
        nr = 1
        months = ('Jan','Feb','Mar','Apr','May','Jun','Jul','Aug','Sep','Oct','Nov','Dec')
        ind = np.arange(len(months))
        
        # no graph for less than 2 scenarios
        if nc < 2:
            return -1
        # bar width
        width = 0.45
        # fig is figure - axarr is vector of plots (many plots in one figure) 
        fig, axarr = plt.subplots(nrows=nr, ncols=nc)
        blues = ['SteelBlue',
                 'SkyBlue',
                 'LightBlue']
        reds = ['IndianRed',
                'Tomato',
                'LightCoral']
        
        # go thru scenarios and plot monthly graph for each scenario
        for k,sc in enumerate(scen_list):
            col1 = itertools.cycle(blues)
            col2 = itertools.cycle(reds)
            # get monthly result data from pickle file
            with open(cpath+'scen_months_'+sc+'.pickle', 'rb') as handle:
                dic = pickle.load(handle)
            # Producer bars
            bot = np.zeros(len(months))
            for i, p in enumerate(sorted(dic['Producer'].keys())):
                x = ind - width/2
                y = dic['Producer'][p]
                axarr[k].bar(x,y,width,color=next(col1),bottom=bot,label = p+'_cool')
                bot+=y
            # Heat supply bars
            bot2 = np.zeros(len(months))
            for i, s in enumerate(sorted(dic['Supply'].keys())):
                x2 = ind + width/2
                y2 = dic['Supply'][s]
                axarr[k].bar(x2,y2,width,color=next(col2),bottom=bot2,label = s+'_heat')
                bot2+=y2
            
            # Y-axis label for first graph only
            if k == 0:
                axarr[k].set_ylabel('Production (kWh)')
            # Graph settings in general
            axarr[k].set_title(sc)
            axarr[k].set_xticks(ind)
            axarr[k].set_xticklabels(months)
            axarr[k].legend()
            for item in ([axarr[k].title, axarr[k].xaxis.label, axarr[k].yaxis.label] + axarr[k].get_xticklabels() + axarr[k].get_yticklabels() + axarr[k].legend().get_texts()):
                item.set_fontsize(8)    
            for item in ([axarr[k].title, axarr[k].xaxis.label, axarr[k].yaxis.label] + axarr[k].get_xticklabels()):
                item.set_fontsize(7)
        # Define figure size (4 inch width per graph and 5.5 inch height)
        fig.set_size_inches(nc*4,5.5)
        fig.subplots_adjust(top=0.92,right=0.98,left=0.08,bottom=0.07,wspace=0.24)
        fig.tight_layout()
        # Show graph window
        plt.show()   
    
    def update_economic_results(self):
        # Get figure and canvas objects
        fig = self.fig4
        canv = self.canvas4
        # Clear figure
        fig.clf()

        dic = self.profit_dic
        res_mat = dic['Case']['CCR']
        if 'Reference' in dic:    
            ref_mat = dic['Reference']['CCR']
        else:
            ref_mat = [1.0]*len(res_mat)
        
        # Create plots
        ind = np.arange(len(res_mat))
        pl = fig.add_subplot(1,1,1)
        width = 0.45
        pl.bar(ind-width/2,res_mat,width,color = 'tomato',label = 'Profitability - Case')
        pl.bar(ind+width/2,ref_mat,width,color = 'mediumaquamarine',label = 'Profitability - Reference')
        pl.set_ylabel('Euros')
        pl.set_title('Profitability of DC')
        pl.set_xticks(ind)
        pl.set_xticklabels(ind)
        pl.legend()
        pl.grid(b=True,color='0.5',linestyle='--',axis = 'y')
        fig.subplots_adjust(top=0.92,right=0.98,left=0.16,bottom=0.07,hspace=0.20)
        for item in ([pl.title, pl.xaxis.label, pl.yaxis.label] + pl.get_xticklabels() + pl.get_yticklabels() + pl.legend().get_texts()):
            item.set_fontsize(8)
        canv.draw()
        
        # Create plots
        dic2 = self.value_dic['Case']
        dic3 = self.value_dic['Reference']
        key_vec = ['Total costs (ke)',
                   'Cooling production costs (e/MWh)']
        tab = self.profit_tab
        tab.clear()
        nc = 3
        nr = len(key_vec)+2
        tab.setColumnCount(nc)
        tab.setRowCount(nr)
        font = QFont()
        font.setBold(True)
        item1 = QTableWidgetItem('Case')
        item2 = QTableWidgetItem('Reference')
        item1.setTextAlignment(Qt.AlignRight)
        item2.setTextAlignment(Qt.AlignRight)
        item1.setFont(font)
        item2.setFont(font)
        tab.setItem(0,1,item1)
        tab.setItem(0,2,item2)
        
        for i, r in enumerate(key_vec):
            item = QTableWidgetItem(r)
            item2 = QTableWidgetItem(str(round((dic2[r]))))
            if dic3[r] == '-':
                item3 = QTableWidgetItem('-')
            else:
                item3 = QTableWidgetItem(str(round((dic3[r]))))
            item2.setTextAlignment(Qt.AlignRight)
            item3.setTextAlignment(Qt.AlignRight)
            tab.setItem(i+1,0,item)
            tab.setItem(i+1,1,item2)
            tab.setItem(i+1,2,item3) 
        item = QTableWidgetItem('Payback period')
        item2 = QTableWidgetItem(str(round(dic['Case']['PBP'])))
        item3 = QTableWidgetItem(str(round(dic['Reference']['PBP'])))  
        tab.setItem(nr-1,0,item)
        tab.setItem(nr-1,1,item2)
        tab.setItem(nr-1,2,item3) 
        item2.setTextAlignment(Qt.AlignRight)
        item3.setTextAlignment(Qt.AlignRight)        
        
        tab.setStyleSheet("QTableWidget {font-size: 14px;}")   
        tab.resizeColumnsToContents()
        tab.resizeRowsToContents()
        tab.horizontalHeader().setVisible(False)
        tab.verticalHeader().setVisible(False)
        tab.clearSpans()
        

    def update_emission_results(self):
        # Get figure and canvas objects
        fig = self.fig5
        canv = self.canvas5
        # Clear figure
        fig.clf()
        blues = ['SteelBlue','SkyBlue','LightBlue']
        reds = ['IndianRed','Tomato','LightCoral']
        width = 0.7
        
        # Calculate total emissions
        dic = self.emi_dic
        tot_emi = [sum(dic['Operational'].values()),sum(dic['Embedded'].values())]
        tot_cat = ['Operational','Embedded']
        ind1 = np.arange(len(tot_cat))
        
        # Get values from emi_dic
        ope_cat = sorted(dic['Operational'].keys())
        ind2 = np.arange(len(ope_cat))
        ope_emi = []
        for c in ope_cat:
            ope_emi.append(dic['Operational'][c])
        emb_cat = sorted(dic['Embedded'].keys())
        emb_cat_wrap = [ '\n'.join(wrap(l, 10)) for l in emb_cat]
        ind3 = np.arange(len(emb_cat))
        emb_emi = []
        for c in emb_cat:
            emb_emi.append(dic['Embedded'][c])
        
        # Create emission plots
        gs = gridspec.GridSpec(2, 2)
        pl1 = fig.add_subplot(gs[0,:])
        pl1.barh(ind1,tot_emi,width,color = 'DeepSkyBlue')
        pl1.set_title('Total emissions (kg CO2 eq)')
        pl1.set_yticks(ind1)
        pl1.set_yticklabels(tot_cat)
        pl1.grid(b=True,color='0.5',linestyle='--',axis = 'x')
        
        pl2 = fig.add_subplot(gs[1,0])
        pl2.bar(ind2,ope_emi,width,color = 'skyblue')
        pl2.set_title('Operation emissions')
        pl2.set_xticks(ind2)
        pl2.set_xticklabels(ope_cat)
        pl2.grid(b=True,color='0.5',linestyle='--',axis = 'y')       
        
        pl3 = fig.add_subplot(gs[1,1])
        pl3.bar(ind3,emb_emi,width,color = 'lightskyblue')
        pl3.set_title('Embedded emissions')
        pl3.set_xticks(ind3)
        pl3.set_xticklabels(emb_cat_wrap)
        pl3.grid(b=True,color='0.5',linestyle='--',axis = 'y')
        
        fig.subplots_adjust(top=0.92,right=0.98,left=0.16,bottom=0.10,hspace=0.35,wspace = 0.30)
        
        for item in ([pl1.title, pl1.xaxis.label, pl1.yaxis.label] + pl1.get_xticklabels() + pl1.get_yticklabels()):
            item.set_fontsize(8)
        for item in ([pl2.title, pl2.xaxis.label, pl2.yaxis.label] + pl2.get_xticklabels() + pl2.get_yticklabels()):
            item.set_fontsize(7)
        for item in ([pl3.title, pl3.xaxis.label, pl3.yaxis.label] + pl3.get_xticklabels() + pl3.get_yticklabels()):
            item.set_fontsize(7) 
        canv.draw()
        
        # Create emission table
        dic2 = self.value_dic['Case']
        key_vec = ['Total emissions (tCO2)',
                   'Operation emissions (tCO2)',
                   'Embedded emissions (tCO2)',
                   'Share of embedded emissions (%)',
                   'Cooling emissions (kgCO2/MWh)']
        tab = self.emi_tab
        tab.clear()
        nc = 2
        nr = len(key_vec)
        tab.setColumnCount(nc)
        tab.setRowCount(nr)
        for c in range(nc):
            for i, r in enumerate(key_vec):
                if c == 0:
                    item = QTableWidgetItem(r)
                    
                else:
                    item = QTableWidgetItem(str(round((dic2[r]))))
                    item.setTextAlignment(Qt.AlignRight)
                tab.setItem(i,c,item)   
        tab.setStyleSheet("QTableWidget {font-size: 14px;}")   
        tab.resizeColumnsToContents()
        tab.resizeRowsToContents()
        tab.horizontalHeader().setVisible(False)
        tab.verticalHeader().setVisible(False)
        tab.clearSpans()
        
    
    # Open utilisation factor table window
    def utilisation_factors(self):
    # if there is open result, proceed
        if self.period > 0:
            # Create utilisation factor table object and show it
            self.af_tab = scen_af(None,self)
            self.af_tab.show()
        
    # Get capacity value data from scenario file for sensitivity analysis
    def get_sensitivity_data(self,case,scen):
        dic = defaultdict(dict)
        case_path = '.\\GIS\\'+case+'\\'
        fname = case_path+'System_data_'+scen+'.txt'
        # read scenario file
        if os.path.exists(fname):
            with open(fname,'r') as fp:
                for f in fp:
                    data_vec = f.strip('\n').split('\t')
                    if len(data_vec) > 1:
                        # Pick capacity values from scenario file lines
                        if data_vec[0][0] == 'P':
                            dic['Producers'][data_vec[0]] = [data_vec[1],float(data_vec[2])]
                        elif data_vec[0][0] == 'T':
                            dic['Storage'][data_vec[0]] = [data_vec[1],float(data_vec[2])]
                        elif data_vec[0][0] == 'S':
                            dic['Supply'][data_vec[0]] = [data_vec[1],float(data_vec[5])]
            return dic
        else: 
            # if file does not exist, return empty dictionary
            dic = {}
            return dic

    # Update unit list in sensitivity analysis when case or scenario are selected in combobox
    def update_sensitivity_table(self):
        case = self.sens_case.currentText()
        scen = self.sens_scen.currentText()
        
        self.sens_dic = self.get_sensitivity_data(case, scen)
        tab = self.sens_unit_list
        tab.clear()
        for d in sorted(self.sens_dic.keys()):
            for k in sorted(self.sens_dic[d].keys()):
                line_str = k+' - '+self.sens_dic[d][k][0]
                tab.addItem(line_str)
        tab.setCurrentRow(0)
    
    # Update capacity value in sensitivity analysis when unit in active list double-clicked    
    def update_capacity_value(self):
        dic = self.sens_dic
        if self.sens_unit_list.count() > 0:
            # Get unit data from list item
            line = self.sens_unit_list.currentItem().text()
            unit = line.split('-')[0].strip(' ')
            case = self.sens_case.currentText()
            scen = self.sens_scen.currentText()
            # Define capacity unit
            if unit[0] == 'P':
                dkey = 'Producers'
                kw = 'kW'
            elif unit[0] == 'S':
                dkey = 'Supply'
                kw = 'kW'
            elif unit[0] == 'T':
                dkey = 'Storage'
                kw = 'kWh'
            desc = dic[dkey][unit][0]
            cap = dic[dkey][unit][1]
            self.sens_case_detail.setText('Scenario: '+case+' - '+scen)
            self.sens_unit_detail.setText('Unit: '+unit+' - '+desc)
            self.cap_label.setText('Capacity: '+str(cap)+' '+kw)
            # Update global variables for run_sensitivity -function
            self.sens_unit['Unit'] = unit
            self.sens_unit['Case'] = case
            self.sens_unit['Scen'] = scen
    
    # sensitivity analysis starts, runs several optimisations and updates result table and graph        
    def run_sensitivity(self):
        # if sensitivity unit (producer, supply or storage) is selected then proceed
        if bool(self.sens_unit):
            # check validity for min or max numbers
            try:
                cmin = float(self.sens_min.text())
                cmax = float(self.sens_max.text()) 
            except ValueError:
                msgBox = QMessageBox()
                msgBox.setText("Incorrect values for boundaries.")
                msgBox.exec_()
                return -1
            # max has to be > min
            if cmax > cmin:
                cap_vec = []
                nod_nr = self.sens_num.value()
                # create vector of capacity values
                diff = (cmax-cmin)/(nod_nr-1)
                for i in range(nod_nr):
                    cap_vec.append(cmin+i*diff)
            else:
                msgBox = QMessageBox()
                msgBox.setText("Incorrect values for boundaries.")
                msgBox.exec_()
                return -1
           
            # Empty graph and table
            fig = self.fig3
            canv = self.canvas3
            fig.clf()
            tab = self.sens_table
            tab.clear()
           
            # Sensitivity result dic
            sens_res = defaultdict(list)
            sens_graf = defaultdict(list)
            # Case details
            case = self.sens_unit['Case']
            case_path = '.\\GIS\\'+case+'\\'            
            scen = self.sens_unit['Scen']
            unit = self.sens_unit['Unit']
            u_typ = unit[0]
            u_idx = unit[1:]
            # Dictionaries initialised
            par_dic = defaultdict(list)
            ts_dic = defaultdict(list)
            nod_dic = {}
            loc_dic = {}
            # read nodes and node locations
            period = 8760
            with open(case_path+'dh_nodes_'+scen+'.txt','r') as ll:
                for l in ll:
                    vec = l.strip('\n').split('\t')
                    loc_dic[vec[0]] = literal_eval(vec[1])
            with open(case_path+'prod_con_'+scen+'.txt','r') as pp:
                for p in pp:
                    vec = p.strip('\n').split('\t')
                    nod_dic[vec[0]] = vec[1]                    
            # Get consumption time series from con_ts.pickle in case folder
            cdic = case_path+'con_ts.pickle'
            # Check if pickle-file exists
            if os.path.exists(cdic):
                with open(cdic, 'rb') as handle:
                    ts_dic = pickle.load(handle)
            else:
                msgBox = QMessageBox()
                msgBox.setText("No consumption timeseries available: con_ts.pickle")
                msgBox.exec_()
                return 0
            
            # Read tariffs from System_cost.txt
            with open(case_path+'System_cost_'+scen+'.txt','r') as sp:
                for s in sp:
                    vec = s.strip('\n').split(' ')
                    if vec[0] == 'Energy':
                        e_tariff = [float(vec[1])]*period
                    elif vec[0] == 'Capacity':
                        c_tariff = float(vec[1])
                    else:
                        location = vec[0] 
            # Create system and initialise
            self.system = System()
            self.system.initialise(location)                 
            # Create cooling system (case name, True if network, energy tariff, capacity tariff)
            self.system.addCoolingSystem(case, True, e_tariff, c_tariff)
            self.system.initNetwork(case,scen)
            # Read component data from System_data.txt
            
            p_indx = 0
            s_indx = 0
            with open(case_path+'System_data_'+scen+'.txt','r') as fp:
                for f in fp:
                    data_vec = f.strip('\n').split('\t')
                    if len(data_vec) > 1:
                        if data_vec[0][0] == 'C':
                            coord = loc_dic[nod_dic[data_vec[0]]]
                            c_indx = data_vec[0][1:]
                            c_type = data_vec[1]
                            demand = ts_dic[c_type][:period]
                            self.system.addConsumer(case,c_indx,demand)
                        elif data_vec[0][0] == 'L':
                            pass
                        elif data_vec[0][0] == 'P':
                            coord = loc_dic[nod_dic[data_vec[0]]]
                            p_indx = data_vec[0][1:]
                            par = []
                            for i in range(2,8):
                                par.append(float(data_vec[i]))
                            self.system.addProducer(case,p_indx,par[0],par[1],par[2],par[3],par[4],par[5])
                        elif data_vec[0][0] == 'T':
                            par = []
                            for i in range(2,7):
                                par.append(float(data_vec[i]))     
                            self.system.defineSimpleStorage(case,data_vec[1],par[0],par[1],par[2],par[3],par[4])
                        elif data_vec[0][0] == 'S':
                            s_indx = data_vec[0][1:]
                            par = []
                            for i in range(3,8):
                                par.append(float(data_vec[i]))
                            self.system.addSupply(case,s_indx,data_vec[1],data_vec[2],par[0],par[1],par[2],par[3],par[4])
            
            # Row headers in table
            heads = ['Total cooling (MWh)',
                     'Total costs (ke)',
                     'Total emissions (tCO2)',
                     'Operation emissions (tCO2)',
                     'Embedded emissions (tCO2)',
                     'Share of embedded emissions (%)',
                     'Cooling production costs (e/MWh)',    
                     'Cooling emissions (kgCO2/MWh)']
            
            # Get current system object
            for s in self.system.systems:
                if s.name == case:
                    sensys = s
            # Clear and update log
            self.sens_log.clear()
            self.sens_log.append('Sensitivity analysis starts:')
            QApplication.processEvents()     
            # Iterate thru capacity vector and optimise
            for i,c in enumerate(cap_vec):
                cap = str(round(c))
                print('Optimisation: '+cap)
                # Update capacity value for selected unit
                if u_typ == 'P':
                    for j in range(0, len(sensys.producers)):
                        if sensys.producers[j].node_id == u_idx:
                            sensys.producers[j].capacity = c
                elif u_typ == 'S':
                    for j in range(0, len(sensys.supply)):
                        if sensys.supply[j].node_id == u_idx:
                            sensys.supply[j].capacity = c
                elif u_typ == 'T':
                    s_type = self.sens_dic['Storage'][unit][0]
                    sensys.storages[s_type].capacity = c
                # Build and run model    
                opt_sta = self.system.runOptimisation(False,case,period)                
                # Get result and write dictionaries for graph (sens_graf) and table (sens_res)

                if i == 0:
                    sens_res['Capacity'] = []
                    for v in heads:
                        sens_res['Capacity'].append(v)
                if opt_sta == 1:
                    value_dic = self.system.getResultsSummary(case,location)
                    for v in heads:
                        sens_res[cap].append(round(value_dic[v],1))
                        sens_graf[v].append(value_dic[v])
                else:
                    for v in heads:
                        sens_res[cap].append('-')
                        sens_graf[v].append(float('nan'))
                self.sens_log.append(str(round(c))+' - optimisation ready')
                QApplication.processEvents()
            self.sens_log.append('Sensitivity analysis ready')
            QApplication.processEvents()
            
            # Update sensitivity table        

            nc = len(sens_res.keys())
            nr = len(sens_res['Capacity'])+1
            tab.setColumnCount(nc)
            tab.setRowCount(nr)
            font = QFont()
            font.setBold(True)
            for c in range(nc):
                for r in range(nr):
                    if c == 0:
                        if r == 0:
                            item = QTableWidgetItem('Capacity')
                            item.setFont(font)
                        else:
                            item = QTableWidgetItem(sens_res['Capacity'][r-1]) 
                    else:
                        if r == 0:
                            item = QTableWidgetItem(str(round(cap_vec[c-1])))
                            item.setFont(font)
                        else:
                            res = str(sens_res[str(round(cap_vec[c-1]))][r-1])
                            item = QTableWidgetItem(res) 
                        item.setTextAlignment(Qt.AlignRight) 
                    tab.setItem(r,c,item)           
            tab.setStyleSheet("QTableWidget {font-size: 14px;}")   
            tab.resizeColumnsToContents()
            tab.resizeRowsToContents()
            tab.clearSpans()
            
            # Get table content size and resize table object
            y = tab.verticalHeader().length()+tab.frameWidth()*2
            x = tab.horizontalHeader().length()+tab.frameWidth()*2
            tab.setFixedSize(min(self.mx,x),y) 
            tab.show()
            

            # Results presented in graph
            p_vec = ['Total costs (ke)','Total emissions (tCO2)'] 
            ax = fig.add_subplot(111)
            col = ['darkgreen',
                   'mediumseagreen',
                   'lawngreen',
                   'limegreen',
                   'lightgreen',
                   'lawngreen',
                   'limegreen',
                   'lightgreen']
            xx = cap_vec
            yy1 = sens_graf[p_vec[0]]
            yy2 = sens_graf[p_vec[1]]
            ax2 = ax.twinx()
            ax.set_ylabel('keuro')
            ax2.set_ylabel('tCO2')
            ax.plot(xx,yy1,label = p_vec[0].strip(' (tCO2)').strip(' (ke)'),color = col[0])
            ax2.plot(xx,yy2,label = p_vec[1].strip(' (tCO2)').strip(' (ke)'),color = col[1])
            
            fig.subplots_adjust(top=0.95,right=0.90,left=0.10,bottom=0.07,hspace=0.20)
            ax.legend(loc=1)
            ax2.legend(loc=2)
            ax.grid(b=True,color='0.5',linestyle='--')
            ax.set_xlim(cap_vec[0],cap_vec[-1])
            canv.draw()
        else:
            return -1   
     
    # update reference case table
    def update_ref_table(self):
        if self.gis_status == 1:
            dic = self.gis.reference_con
            ric = self.gis.resource_dic
            tsd = self.ts_dic
            # get names of cooling tecs
            ctecs = sorted(self.gis.cool_dic.keys())
            abs_eff = float(self.gis.cool_dic['Absorption'][3])
            tab = self.ref_table
            tab.clear()
            row_heading = sorted(dic.keys())
            col_heading = ['Consumer type   ',
                           'Cooling method  ',
                           'Cooling capacity',
                           'Heating method  ',
                           'Heating fuel    ',
                           'Heating capacity',
                           'Solar option    ',
                           'Solar capacity  ']
            nr = len(row_heading)
            nc = len(col_heading)
            tab.setColumnCount(nc)
            tab.setRowCount(nr)
            tab.setHorizontalHeaderLabels(col_heading)
            tab.setVerticalHeaderLabels(row_heading)
            combo_vec  = []
            combo_vec2 = []
            heat_tecs = ['Boiler','CHP']
            combo_vec3 = []
            fuel_vec = ric['CHP']
            combo_vec4 = []
            solar_vec = ['No','Collectors','PVs']
            font = QFont()
            font.setPixelSize(14) 
            
            # Create columns for reference data
            for i, r in enumerate(row_heading):
                for c in range(nc):
                    if c == 0:
                        item = QTableWidgetItem(dic[r][0])
                        tab.setItem(i,c,item)
                    elif c == 1:                        
                        combo_vec.append(QComboBox())
                        for tt in ctecs:
                            combo_vec[i].addItem(tt)
                        combo_vec[i].setFont(font)
                        tab.setCellWidget(i,c,combo_vec[i])
                        combo_vec[i].setCurrentIndex(dic[r][1])
                    elif c == 2:
                        max_val = max(tsd[dic[r][0]])
                        item = QTableWidgetItem(str(max_val))
                        tab.setItem(i,c,item)
                    elif c == 3:
                        combo_vec2.append(QComboBox())
                        for ht in heat_tecs:
                            combo_vec2[i].addItem(ht)
                        combo_vec2[i].setFont(font)
                        tab.setCellWidget(i,c,combo_vec2[i])
                        combo_vec2[i].setCurrentIndex(dic[r][3])
                    elif c == 4:
                        combo_vec3.append(QComboBox())
                        for ft in fuel_vec:
                            combo_vec3[i].addItem(ft)
                        combo_vec3[i].setFont(font)
                        tab.setCellWidget(i,c,combo_vec3[i])
                        combo_vec3[i].setCurrentIndex(dic[r][4])
                    elif c == 5:
                        val = max_val*abs_eff
                        item = QTableWidgetItem(str(val))
                        tab.setItem(i,c,item)
                    elif c == 6:
                        combo_vec4.append(QComboBox())
                        for st in solar_vec:
                            combo_vec4[i].addItem(st)
                        combo_vec4[i].setFont(font)
                        tab.setCellWidget(i,c,combo_vec4[i])
                        combo_vec4[i].setCurrentIndex(dic[r][6])    
                    else:
                        val = dic[r][7]
                        item = QTableWidgetItem(str(val))
                        tab.setItem(i,c,item)
            tab.setStyleSheet("QTableWidget {font-size: 14px;}")
            tab.horizontalHeader().setStyleSheet("QHeaderView {font-size: 14px;}")
            tab.verticalHeader().setStyleSheet("QHeaderView {font-size: 14px;}")
            # resize rows and columns to fit contents and remove spans
            tab.resizeColumnsToContents()
            tab.resizeRowsToContents()
            tab.clearSpans()
            tab.setVerticalScrollBarPolicy(Qt.ScrollBarAlwaysOff)
            self.ref_tab_status = 1
    
    # update reference case dictionary reference_con
    def update_ref_dic(self):
        if self.ref_tab_status == 1:
            tab = self.ref_table
            dic = self.gis.reference_con
            rn = tab.rowCount()
            cn = tab.columnCount()
            for r in range(rn):
                unit = tab.verticalHeaderItem(r).text()
                vec = []
                for c in range(cn):
                    if c == 0:
                        val = tab.item(r,c).text()
                        vec.append(val)
                    elif c in [1,3,4,6]:
                        val = tab.cellWidget(r,c).currentIndex()
                        vec.append(val)
                    else:
                        val = float(tab.item(r,c).text())
                        vec.append(val)
                dic[unit] = vec

    
    # update indicator result table for scenario        
    def update_res_table(self):
        if self.period > 0:
            dic = self.value_dic['Case']
            key_vec = sorted(dic.keys())
            tab = self.annu_table
            nc = 2
            nr = len(key_vec)
            tab.setColumnCount(nc)
            tab.setRowCount(nr)
            for c in range(nc):
                for i, r in enumerate(key_vec):
                    if c == 0:
                        item = QTableWidgetItem(r)
                        
                    else:
                        item = QTableWidgetItem(str(round((dic[r]))))
                        item.setTextAlignment(Qt.AlignRight)
                    tab.setItem(i,c,item)   
            tab.setStyleSheet("QTableWidget {font-size: 14px;}")   
            tab.resizeColumnsToContents()
            tab.resizeRowsToContents()
            tab.clearSpans()
     
    # update utilisation factor table           
    def update_af_table(self):
        if self.period > 0:
            tab = self.af_table
            tab.clear()
            adic = self.af_dic
            tdic = self.unit_type
            nc = 2
            nr = len(tdic.keys())+3
            tab.setColumnCount(nc)
            tab.setRowCount(nr)
            r = 0
            font = QFont()
            font.setBold(True)
            for k in sorted(adic.keys()):
                if k == 'Storage':
                    kaf = k+' (nr of cycles)'
                else:
                    kaf = k+' (utilisation rate)'
                item = QTableWidgetItem(kaf)
                item.setFont(font)
                tab.setItem(r,0,item)
                r+=1
                for i in sorted(adic[k].keys()):
                    if i in list(tdic.keys()):
                        if i == tdic[i]:
                            item0 = QTableWidgetItem(i)
                            item1 = QTableWidgetItem(str(round(adic[k][i])))
                            item1.setTextAlignment(Qt.AlignRight)
                        else:
                            item0 = QTableWidgetItem(i+' - '+tdic[i])
                            item1 = QTableWidgetItem(str(round(adic[k][i],2)))
                            item1.setTextAlignment(Qt.AlignRight)
                        tab.setItem(r,0,item0)
                        tab.setItem(r,1,item1)
                        r+=1        
            tab.setStyleSheet("QTableWidget {font-size: 14px;}")   
            tab.resizeColumnsToContents()
            tab.resizeRowsToContents()
            tab.clearSpans()        
            tab.horizontalHeader().setVisible(False)
            tab.verticalHeader().setVisible(False)
            tab.setVerticalScrollBarPolicy(Qt.ScrollBarAlwaysOff)
            tab.setHorizontalScrollBarPolicy(Qt.ScrollBarAlwaysOff)  
                             
    # updates data table when unit selection (consumers, producers etc...) combobox activated
    def update_data_table(self):
        data_type = self.unit_selection.currentText()
        if self.gis_status == 1:
            if data_type == 'Consumers':
                self.tot_cons.show()
            else:
                self.tot_cons.hide()
            dic = self.gis.archive_dic[data_type]
            tab = self.unit_par_table
            tab.clear()
            col_heading = dic['desc']
            row_heading = sorted(dic.keys())
            nc = len(col_heading)
            nr = len(row_heading)-1
            tab.setColumnCount(nc)
            tab.setRowCount(nr)
            for c in range(nc):
                for i, r in enumerate(row_heading):
                    if r != 'desc':
                        item = QTableWidgetItem(dic[r][c])
                        tab.setItem(i,c,item)
            tab.setHorizontalHeaderLabels(col_heading)
            tab.setVerticalHeaderLabels(row_heading)
            tab.setStyleSheet("QTableWidget {font-size: 14px;}")
            tab.horizontalHeader().setStyleSheet("QHeaderView {font-size: 14px;}")
            tab.verticalHeader().setStyleSheet("QHeaderView {font-size: 14px;}")
            # resize rows and columns to fit contents and remove spans
            tab.resizeColumnsToContents()
            tab.resizeRowsToContents()
            tab.clearSpans()
            tab.setVerticalScrollBarPolicy(Qt.ScrollBarAlwaysOff)
            #tab.setHorizontalScrollBarPolicy(Qt.ScrollBarAlwaysOff)           
    
    # save modifications in data table into archive_dic dictionary
    def save_data_table(self):
        unit = self.unit_selection.currentText()
        if self.gis_status == 1:
            tab = self.unit_par_table
            dic = self.gis.archive_dic[unit]
            rn = tab.rowCount()
            cn = tab.columnCount()
            for r in range(rn):
                for c in range(cn):
                    comp = tab.verticalHeaderItem(r).text()
                    val = tab.item(r,c).text()
                    dic[comp][c] = val
    
    def show_total_consumption(self):
        ts = self.ts_dic
        if bool(ts):
            
            dic = self.gis.archive_dic['Consumers']
            dkey = sorted(dic.keys())
            dkey.remove('desc')
            if len(dkey) > 0:
                for i,d in enumerate(dkey):
                    if i == 0:
                        t = np.arange(len(ts[dic[d][0]]))
                        con_y = np.array(ts[dic[d][0]])
                    else:
                        con_y += np.array(ts[dic[d][0]])
                fig, ax = plt.subplots()
                y = -np.sort(-con_y)
                ax.plot(t,y,'b-',label='Total consumption - duration curve', linewidth=3, color = 'seagreen')
                ax.grid(b=True,color='0.5',linestyle='--')
                ax.legend()
                fig.subplots_adjust(top=0.92,right=0.98,left=0.08,bottom=0.07,wspace=0.24)
                plt.show()
            else:
                return 0
        
    def show_duration_curve(self):
        cd = self.cap_dic
        
        if bool(cd):
            pd = self.result_prod_dic
            sd = self.result_sup_dic
            td = self.result_sto_dic
            
            col_p = ['darkgreen',
                     'lawngreen',
                     'mediumseagreen',
                     'limegreen',
                     'lightgreen']
            col_s = ['indigo',
                     'darkorchid',
                     'mediumorchid',
                     'plum',
                     'pink'] 
            col_t = ['navy',
                     'royalblue',
                     'dodgerblue',
                     'skyblue'] 
            # 'District cooling'
            # 'Heat'
            pk = sorted(pd.keys())
            sk = sorted(sd.keys())
            tk = sorted(td.keys())
            fig, ax = plt.subplots()
            col = itertools.cycle(col_p)
            for p in pk:
                cap = cd['Producer'][p]
                if cap>0:
                    y = -np.sort(-np.array(pd[p]['District cooling'])/cap)
                    t = np.arange(len(y))
                    ax.plot(t,y,color=next(col),label='DC production - '+p,linewidth=3)
                    ax.legend()
            col = itertools.cycle(col_s)
            for s in sk:
                cap = cd['Supply'][s]
                if cap>0:
                    y = -np.sort(-np.array(sd[s]['Heat'])/cap)
                    t = np.arange(len(y))
                    ax.plot(t,y,color=next(col),label='Heat supply - '+s,linewidth=3)
                    ax.legend()
            '''
            col = itertools.cycle(col_t)
            for t in tk:
                if t in list(cd['Storage'].keys()):
                    cap = cd['Storage'][t]
                    if cap>0:
                        y = -np.sort(-np.array(td[t]))
                        t = np.arange(len(y))
                        ax.plot(t,y,color=next(col),label='Storage - '+t,linewidth=3)
            '''
            ax.grid(b=True,color='0.5',linestyle='--')                    
            fig.subplots_adjust(top=0.92,right=0.98,left=0.08,bottom=0.07,wspace=0.24)
            
            plt.show()
        
    
    # import time series from csv-files
    def import_ts(self,dic):
        if dic == self.ts_dic:
            fname = self.ts_name.text()
            self.ts_name.setText('')
        else:
            fname = self.ts_name_2.text()
            self.ts_name_2.setText('')
        if self.gis_status == 1:
            if fname:
                for d in list(dic.keys()):
                    del dic[d]
                self.read_csv(fname, dic)
                if dic == self.ts_dic:
                    self.update_con_ts_table()
                    self.gis.con_type.clear()
                    dvec = sorted(self.ts_dic.keys())
                    for d in dvec:
                        self.gis.con_type.addItem(d)
                    # if consumer has new consumer profile it is not deleted
                    for k in list(self.gis.gview.cons_db.keys()):
                        code = self.gis.gview.cons_db[k][0]
                        prof = self.gis.gview.cons_db[k][2]
                        if prof not in dvec:
                            self.gis.gview.scene.removeItem(k)
                            del self.gis.gview.cons_db[k]
                            del self.gis.archive_dic['Consumers'][code]
                            del self.gis.reference_con[code]                       
                if dic == self.res_dic:
                    self.update_res_ts_table()
            
    # update consumer profile time series table in parameters tab       
    def update_con_ts_table(self):
        if self.gis_status == 1:
            dic = self.ts_dic
            tab = self.ts_table
            col_heading = ['Code','Points','Average','Max','Min']
            nc = len(col_heading)
            nr = len(dic.keys())
            tab.setColumnCount(nc)
            tab.setRowCount(nr)
            for c in range(nc):
                for r, k in enumerate(sorted(dic.keys())):
                    if c == 0:
                        val = k
                    elif c == 1:
                        val = str(len(dic[k]))
                    elif c == 2:
                        if len(dic[k]) > 0:
                            val_s = 0
                            for i in dic[k]:
                                val_s+=i
                            val = str(round(val_s/len(dic[k]),2))
                        else:
                            val = '-'
                    elif c == 3:
                        if len(dic[k]) > 0:
                            val = str(max(dic[k]))                         
                        else:
                            val = '-'
                    elif c == 4:
                        if len(dic[k]) > 0:
                            val = str(min(dic[k]))
                        else:
                            val = '-'
                    item = QTableWidgetItem(val)
                    tab.setItem(r,c,item)
            tab.setHorizontalHeaderLabels(col_heading)
            tab.setStyleSheet("QTableWidget {font-size: 14px;}")
            tab.horizontalHeader().setStyleSheet("QHeaderView {font-size: 14px;}")
            tab.verticalHeader().hide()
            # resize rows and columns to fit contents and remove spans
            tab.resizeColumnsToContents()
            tab.resizeRowsToContents()
            tab.clearSpans()
            #tab.setVerticalScrollBarPolicy(Qt.ScrollBarAlwaysOff)
            #tab.setHorizontalScrollBarPolicy(Qt.ScrollBarAlwaysOff)   
    
    # update resource data time series table in parameters tab     
    def update_res_ts_table(self):
        # gis status indicates if gis window is active ie. case is being under editing
        if self.gis_status == 1:
            dic = self.res_dic
            tab = self.ts_table_2
            col_heading = ['Resource','Points','Average','Max','Min']
            nc = len(col_heading)
            nr = len(dic.keys())
            tab.setColumnCount(nc)
            tab.setRowCount(nr)
            for c in range(nc):
                for r, k in enumerate(sorted(dic.keys())):
                    if c == 0:
                        val = k
                    elif c == 1:
                        val = str(len(dic[k]))
                    elif c == 2:
                        if len(dic[k]) > 0:
                            val_s = 0
                            for i in dic[k]:
                                val_s+=float(i)
                            val = str(round(val_s/len(dic[k]),2))
                        else:
                            val = '-'
                    elif c == 3:
                        if len(dic[k]) > 0:
                            val = str(max(dic[k]))
                        else:
                            val = '-'
                    elif c == 4:
                        if len(dic[k]) > 0:
                            val = str(min(dic[k]))
                        else:
                            val = '-'
                    item = QTableWidgetItem(val)
                    tab.setItem(r,c,item)
            tab.setHorizontalHeaderLabels(col_heading)
            tab.setStyleSheet("QTableWidget {font-size: 14px;}")
            tab.horizontalHeader().setStyleSheet("QHeaderView {font-size: 14px;}")
            tab.verticalHeader().hide()
            # resize rows and columns to fit contents and remove spans
            tab.resizeColumnsToContents()
            tab.resizeRowsToContents()
            tab.clearSpans()
                  
    
    # open selection window for loading case
    def load_gis_case(self):
        self.load_window = case_selection(parent=None,qwid=self)
        self.load_window.show()
        
    # Save case into .pickle files
    def save_gis_case(self):
        if isinstance(self.gis, QWidget):
            dic1 = self.gis.gview.prod_db
            dic2 = self.gis.gview.line_db
            dic3 = self.gis.gview.cons_db
            dic4 = self.gis.archive_dic
            dic5 = self.ts_dic
            dic6 = self.res_dic
            dic7 = self.gis.reference_con
            val_check = self.gis.export_nodes('BASE')
            if val_check == -1:
                msgBox = QMessageBox()
                msgBox.setText("Network incomplete - case not saved")
                msgBox.exec_()
                return            
            with open(self.gis.gview.path+'prod.pickle', 'wb') as handle:
                pickle.dump(dic1, handle, protocol=pickle.HIGHEST_PROTOCOL)
            with open(self.gis.gview.path+'lines.pickle', 'wb') as handle:
                pickle.dump(dic2, handle, protocol=pickle.HIGHEST_PROTOCOL)  
            with open(self.gis.gview.path+'cons.pickle', 'wb') as handle:
                pickle.dump(dic3, handle, protocol=pickle.HIGHEST_PROTOCOL)
            with open(self.gis.gview.path+'arc.pickle', 'wb') as handle:
                pickle.dump(dic4, handle, protocol=pickle.HIGHEST_PROTOCOL)   
            with open(self.gis.gview.path+'con_ts.pickle', 'wb') as handle:
                pickle.dump(dic5, handle, protocol=pickle.HIGHEST_PROTOCOL)                 
            with open(self.gis.gview.path+'res_ts.pickle', 'wb') as handle:
                pickle.dump(dic6, handle, protocol=pickle.HIGHEST_PROTOCOL)
            with open(self.gis.gview.path+'ref.pickle', 'wb') as handle:
                pickle.dump(dic7, handle, protocol=pickle.HIGHEST_PROTOCOL)
            with open(self.gis.gview.path+'\\System_cost_BASE.txt','w') as f:
                f.write('Energy '+self.etariff.text()+'\n')
                f.write('Capacity '+self.ctariff.text()+'\n')
                f.write(self.loc_combo.currentText())
    
    # update scenario list in analysis tab
    def update_scenario_list(self,typ):
        if typ == 0:
            pname = self.case_combo.currentText()
            path = '.\\GIS\\'+pname+'\\'
            sce_list = self.get_scenario_names(path)
            self.case_list.clear()
            if sce_list != -1:
                self.case_list.addItem('BASE')
                for s in sce_list:
                    self.case_list.addItem(s)
        else:
            pname = self.sens_case.currentText()
            path = '.\\GIS\\'+pname+'\\'
            sce_list = self.get_scenario_names(path)
            self.sens_scen.clear()
            self.sens_unit_list.clear()
            if sce_list != -1:
                self.sens_scen.addItem('BASE')
                for s in sce_list:
                    self.sens_scen.addItem(s)
                self.update_sensitivity_table()
        
    # Save scenario data in scenario files
    def save_gis_scenario(self):
        if self.gis_status == 1:
            # Ask for scenario name
            text,ok = QInputDialog.getText(self,'Scenario','Set scenario name:',QLineEdit.Normal)            
            # Ok pressed
            if ok:
                # Text cannot be empty
                if text != '':
                    # Replace _ with -
                    text.replace('_','-')
                    # Export data
                    val_check = self.gis.export_nodes(text)
                    if val_check == -1:
                        msgBox = QMessageBox()
                        msgBox.setText("Network incomplete - scenario not saved")
                        msgBox.exec_()
                        return
                    else:
                        with open(self.gis.gview.path+'\\System_cost_'+text+'.txt','w') as f:
                            f.write('Energy '+self.etariff.text()+'\n')
                            f.write('Capacity '+self.ctariff.text()+'\n')
                            f.write(self.loc_combo.currentText())
                        # Update scenario list
                        self.update_scenario_list(0)
    
    # Get list of scenario names in case folder
    def get_scenario_names(self,path): 
        flist = os.listdir(path)
        f2 = []
        for f in flist:
            f1 = f.split('.')
            if f1[-1] == 'txt':
                vec = f1[-2].split('_')
                if len(vec) > 1:
                    f2.append(vec[-1])
        f3 = list(set(f2))
        # Remove BASE
        if len(f3) > 0:
            f3.remove('BASE')
            return f3
        else:
            return -1
    
    # Open import case window                        
    def import_gis_case(self):
        self.imp_window = case_import(parent=None)
        self.imp_window.show()
    
    # Open explorer for time series txt files
    def browse_ts(self,line_comp):
        f_name = str(QFileDialog.getOpenFileName(self,filter= "Txt files (*.txt)")[0])
        line_comp.setText(f_name)
        
    # Set resources country file name    
    def set_res_file(self):
        land = self.loc_combo.currentText()
        self.ts_name_2.setText('.\\Data\\resources_'+land+'.txt')
            
    # Read simple csv-file or txt file (header in first row, delimited with ;) into dictionary   
    def read_csv(self,fname,dic):
        with open(fname) as f:
            reader = csv.DictReader(f, delimiter='\t') 
            for row in reader: 
                for (k,v) in row.items(): 
                    dic[k].append(float(v))    
    
    # Makes a copy of current edited case
    def make_case_copy(self):
        # Only if case is loaded
        if self.gis_status == 1:
            # Get list of case folders
            case = self.gis.gview.path
            dir_list = next(os.walk('.\\GIS'))[1]
            # Ask for new case folder name
            text,ok = QInputDialog.getText(self,'Scenario','Set new case folder name:',QLineEdit.Normal) 
            if ok:
                if text in dir_list:
                    msgBox = QMessageBox()
                    msgBox.setText("Case folder exists already - select different name")
                    msgBox.exec_()
                    return 0
                else:
                    # Copy folder to new folder
                    copy_tree(case, '.\\GIS\\'+text)

    def delete_scenarios(self):
        # Only if case is loaded
        if self.gis_status == 1:
            # Get list of case files
            case_path = self.gis.gview.path
            flist = os.listdir(case_path)
            scen_list = []
            # Select files with scenario suffix
            for f in flist:
                vec = f.split('.')[0].split('_')
                if len(vec) > 2:
                    scen_list.append(vec[-1])
            scen_list = list(set(scen_list))
            scen_list.remove('BASE')
            # Only if there are scenarios available
            if len(scen_list) < 1:
                msgBox = QMessageBox()
                msgBox.setText("No scenarios in case folder")
                msgBox.exec_()
                return 0
            # Scenario list items
            scen,ok = QInputDialog.getItem(self,'Scenario','Select scenario:',scen_list)
            if ok:
                for f in flist:
                    vec = f.split('.')[0].split('_')
                    if len(vec) > 2:
                        if vec[-1] == scen:
                            try:
                                os.remove(case_path+'\\'+f)
                            except:
                                pass
            else:
                return 0

# Window for selecting case in load case         
class case_selection(QWidget):
    def __init__(self,parent=None,qwid=None):
        super(case_selection,self).__init__(parent) 
        self.setWindowTitle('Load case')
        self.dir_list = next(os.walk('.\\GIS'))[1]
        self.case_list = QListWidget()
        self.case_list.setFont(QtGui.QFont('SansSerif',12))
        self.main_widget = qwid
        for i in self.dir_list:
            self.case_list.addItem(i)
        self.vb = QVBoxLayout()
        self.hb = QHBoxLayout()
        self.ok_b = QPushButton('OK')
        self.ca_b = QPushButton('Cancel')
        self.ok_b.setFont(QtGui.QFont('SansSerif',12))
        self.ca_b.setFont(QtGui.QFont('SansSerif',12))
                
        self.hb.addWidget(self.ok_b)
        self.hb.addWidget(self.ca_b)
        self.vb.addWidget(self.case_list)
        self.vb.addLayout(self.hb)
        self.setLayout(self.vb)
        
        # if ok update gis, else close window
        self.ok_b.clicked.connect(lambda: self.update_gis())
        self.ca_b.clicked.connect(lambda: self.cancel_window())
    
    # Update gis window for selected case    
    def update_gis(self):
        list_indx = self.case_list.currentRow()
        folder = self.dir_list[list_indx]
        for widget in self.main_widget.gis_widget.children():
            if isinstance(widget, QWidget):
                widget.close()
        self.main_widget.gis = Gis_app(self.main_widget.gis_widget,self.main_widget,folder)
        self.main_widget.gis_status = 1
        self.main_widget.ref_tab_status = 0
        self.main_widget.ref_table.setRowCount(0)
        self.main_widget.gis.gview.load_struct()  
        cdic = self.main_widget.gis.gview.path+'con_ts.pickle'
        if os.path.exists(cdic):
            with open(cdic, 'rb') as handle:
                self.main_widget.ts_dic = pickle.load(handle)
            self.main_widget.update_con_ts_table()
            for d in sorted(self.main_widget.ts_dic.keys()):
                self.main_widget.gis.con_type.addItem(d)    
        else:
            self.main_widget.ts_table.setRowCount(0)
            self.main_widget.gis.con_type.clear()
            self.main_widget.ts_dic.clear()
            #self.main_widget.ts_dic['a'] = 1
        rdic = self.main_widget.gis.gview.path+'res_ts.pickle'
        if os.path.exists(rdic):
            with open(rdic, 'rb') as handle:
                self.main_widget.res_dic = pickle.load(handle)
            self.main_widget.update_res_ts_table()
        else:
            self.main_widget.ts_table_2.setRowCount(0)
            self.main_widget.res_dic.clear()
            #self.main_widget.res_dic['b'] = 1       
        self.main_widget.case_name.setText('Case: '+folder)        
        self.main_widget.update_data_table()
        tfil = self.main_widget.gis.gview.path+'System_cost_BASE.txt'
        if os.path.exists(tfil):
            with open(tfil, 'r') as fp:
                for f in fp:
                    vec = f.strip('\n').split(' ')
                    if vec[0] == 'Energy':
                        self.main_widget.etariff.setText(vec[1])
                    elif vec[0] == 'Capacity':
                        self.main_widget.ctariff.setText(vec[1])
                    else:
                        text = vec[0]
                        index = self.main_widget.loc_combo.findText(text, Qt.MatchFixedString)
                        if index >= 0:
                            self.main_widget.loc_combo.setCurrentIndex(index)
        else:
            self.main_widget.etariff.setText('')
            self.main_widget.ctariff.setText('')
        self.close()
        
    def cancel_window(self):
        self.close()

# Case importer
class case_import(QWidget,Ui_Form):
    def __init__(self,parent=None):
        super(case_import,self).__init__(parent)
        self.setupUi(self)
        self.dir_list = next(os.walk('.\\GIS'))[1]
        self.select_file.clicked.connect(lambda: self.browse_file())
        self.import_but.clicked.connect(lambda: self.process_import())
        self.cancel_but.clicked.connect(lambda: self.exit_import())
        self.warnings.setStyleSheet('color: red')
    
    # Import case data and create case folder    
    def process_import(self):
        case = self.case_name.text()
        path = self.file_name.text()
        lat1 = self.lat1.value()
        lat2 = self.lat2.value()
        lon1 = self.long1.value()
        lon2 = self.long2.value()
        new_path = '.\\GIS\\'+case
        # Case name mandatory
        if not case:
            self.warnings.setText('Case name missing')
            return -1
        if os.path.isdir(new_path):
            self.warnings.setText('Case folder already exists')
            return -1
        # Longitudes and latitudes has to be consistent         
        elif lat1<=lat2 or lon1>=lon2:
            self.warnings.setText('Coordinates not valid')
            return -1 
        # Everything ok
        else:
            # Create case folder            
            os.makedirs(new_path,exist_ok=True)
            # Copy map figure file inot folder
            try:
                copy2(path,new_path+'\\Map.png')
            except:
                self.warnings.setText('Map file copying failed')
                return -1 
            # Write coordinates to txt file
            with open(new_path+'\\Coordinates.txt','w') as f:
                f.write('Latitude1 '+str(lat1)+'\n')
                f.write('Longitude1 '+str(lon1)+'\n')
                f.write('Latitude2 '+str(lat2)+'\n')          
                f.write('Longitude2 '+str(lon2)+'\n')
            # Create default system cost file
            with open(new_path+'\\System_cost_BASE.txt','w') as f:
                f.write('Energy 50\n')
                f.write('Capacity 50\n')
                f.write('Finland') 
        self.close()               
        
    # Browse map png file
    def browse_file(self):
        f_name = str(QFileDialog.getOpenFileName(self,filter= "Image Files (*.png)")[0])
        
        self.file_name.setText(f_name)
    
    # Exit    
    def exit_import(self):
        self.close()


        
# Window for selecting scenario in load results        
class scenario_result(QWidget):
    def __init__(self,parent=None,qwid=None,folder=None):
        super(scenario_result,self).__init__(parent)
        self.setWindowTitle('Load result')
        
        
        self.case_list = QListWidget()
        self.case_list.setFont(QtGui.QFont('SansSerif',12))
        self.main_widget = qwid
        self.cpath = folder
        self.case = self.cpath.strip('\\').split('\\')[-1]
        self.scen_list = []
        
        # Find correct pickle files in file list of case folder
        flist = os.listdir(self.cpath)
        for f in flist:
            vec = f.split('.')
            if len(vec) > 1:
                if vec[1] == 'pickle':
                    if vec[0][:12] == 'scen_results':
                        scen = vec[0].split('_')[-1]
                        self.case_list.addItem(scen)
                        self.scen_list.append(scen)
        # Layout stuff
        self.vb = QVBoxLayout()
        self.hb = QHBoxLayout()
        self.ok_b = QPushButton('OK')
        self.ca_b = QPushButton('Cancel')
        self.ok_b.setFont(QtGui.QFont('SansSerif',12))
        self.ca_b.setFont(QtGui.QFont('SansSerif',12))
        self.hb.addWidget(self.ok_b)
        self.hb.addWidget(self.ca_b)
        self.vb.addWidget(self.case_list)
        self.vb.addLayout(self.hb)
        self.setLayout(self.vb)
        
        self.ok_b.clicked.connect(lambda: self.update_res())
        self.ca_b.clicked.connect(lambda: self.cancel_window())
    
    # Update results for selected scenario
    def update_res(self):
        # There has to be a scenario list
        if self.case_list.currentRow() > -1:
            scen = self.scen_list[self.case_list.currentRow()]
            # Update information about current case and scenario
            self.main_widget.current_scen = scen
            self.main_widget.current_case = self.case
            # Load results
            self.main_widget.load_result_file(self.cpath,scen)
        self.close()
        
    def cancel_window(self):
        self.close()    

# Window for scenario result values (indicators)
class scen_values(QMainWindow):
    def __init__(self,parent=None,qwid=None,folder=None):
        super(scen_values,self).__init__(parent)
        self.setWindowTitle('Result summary')
        self.main_widget = qwid
        self.cpath = folder
        self.scen_list = []
        tab = QTabCustom()
        flist = os.listdir(self.cpath)
        
        # Find correct pickle files in file list of case folder
        for f in flist:
            vec = f.split('.')
            if len(vec) > 1:
                if vec[1] == 'pickle':
                    if vec[0][:11] == 'scen_values':
                        scen = vec[0].split('_')[-1]
                        self.scen_list.append(scen)
        # There has to be scenarios
        if len(self.scen_list) > 0:
            nc = len(self.scen_list)+1
            tab.setColumnCount(nc)
            for k,sc in enumerate(self.scen_list):
                with open(self.cpath+'scen_values_'+sc+'.pickle', 'rb') as handle:
                    dic = pickle.load(handle)
                if k == 0:
                    key_vec = sorted(dic['Case'].keys())
                    nr = len(key_vec)+1
                    tab.setRowCount(nr)
                    for r in range(nr):
                        if r == 0:
                            item = QTableWidgetItem('')
                            tab.setItem(r,0,item)
                        else:
                            item = QTableWidgetItem(key_vec[r-1])
                            tab.setItem(r,0,item)
                c = k+1
                item = QTableWidgetItem(sc)
                item.setTextAlignment(Qt.AlignRight)
                tab.setItem(0,c,item)
                for r,dk in enumerate(sorted(dic['Case'].keys())):                
                    item = QTableWidgetItem(str(round((dic['Case'][dk]))))
                    item.setTextAlignment(Qt.AlignRight)
                    tab.setItem(r+1,c,item)   
            tab.setStyleSheet("QTableWidget {font-size: 14px;}")   
            tab.resizeColumnsToContents()
            tab.resizeRowsToContents()
            tab.clearSpans()        
            tab.horizontalHeader().setVisible(False)
            tab.verticalHeader().setVisible(False)            
            tab.setVerticalScrollBarPolicy(Qt.ScrollBarAlwaysOff)
            tab.setHorizontalScrollBarPolicy(Qt.ScrollBarAlwaysOff)  
            self.setCentralWidget(tab)            
            
            # Resize window to match table
            y = tab.verticalHeader().length()+tab.frameWidth()*2
            x = tab.horizontalHeader().length()+tab.frameWidth()*2
            self.setFixedSize(x,y)   

# Window for utilisation factors
class scen_af(QMainWindow):
    def __init__(self,parent=None,qwid=None):
        super(scen_af,self).__init__(parent)
        self.setWindowTitle('Utilisation factors')
        self.main_widget = qwid
        tab = QTableWidget()
        # Dics for utilisation factors and unit types
        adic = self.main_widget.af_dic
        tdic = self.main_widget.unit_type
        # Tab size definition
        nc = 2
        nr = len(tdic.keys())+3
        tab.setColumnCount(nc)
        tab.setRowCount(nr)
        r = 0
        font = QFont()
        font.setBold(True)
        for k in sorted(adic.keys()):
            item = QTableWidgetItem(k)
            item.setFont(font)
            tab.setItem(r,0,item)
            r+=1
            for i in sorted(adic[k].keys()):
                if i in list(tdic.keys()):
                    if i == tdic[i]:
                        item0 = QTableWidgetItem(i)
                        item1 = QTableWidgetItem(str(round(adic[k][i])))
                    else:
                        item0 = QTableWidgetItem(i+' - '+tdic[i])
                        item1 = QTableWidgetItem(str(round(adic[k][i],2)))
                    tab.setItem(r,0,item0)
                    tab.setItem(r,1,item1)
                    r+=1        
        tab.setStyleSheet("QTableWidget {font-size: 16px;}")   
        tab.resizeColumnsToContents()
        tab.resizeRowsToContents()
        tab.clearSpans()        
        tab.horizontalHeader().setVisible(False)
        tab.verticalHeader().setVisible(False)
        tab.setVerticalScrollBarPolicy(Qt.ScrollBarAlwaysOff)
        tab.setHorizontalScrollBarPolicy(Qt.ScrollBarAlwaysOff)  
        self.setCentralWidget(tab)
        
        y = tab.verticalHeader().length()+tab.frameWidth()*2
        x = tab.horizontalHeader().length()+tab.frameWidth()*2
        self.setFixedSize(x,y)

# Update Navigation toolbar class to include merely 4 important buttons         
class NavigationToolbar(NavToolBar):
    # only display the buttons we need
    toolitems = [t for t in NavToolBar.toolitems if
                 t[0] in ('Home', 'Pan', 'Zoom', 'Save')]

# Table class with copy-paste of entire table content
class QTabCustom(QTableWidget):
    def __init__(self,parent=None):
        super().__init__(parent)
    
    def keyPressEvent(self, event):
        key = event.key()        
        if key == Qt.Key_C:
            modifiers = QApplication.keyboardModifiers()
            if modifiers == Qt.ControlModifier:
                self.copy()
        else:
            QTableWidget.keyPressEvent(self, event)
            
    def copy(self):
        nro = self.rowCount()
        nco = self.columnCount()
        text = '\t'
        for col in range(nco):
            item = self.horizontalHeaderItem(col)
            if item:
                text += item.text()
            else:
                text += str(col)
            if col < nco-1:
                text += '\t'

        text += '\n'    
        for row in range(nro):
            item = self.verticalHeaderItem(row)
            if item:
                text += item.text()
            else:
                text += str(row)
            text += '\t'
            for col in range(nco):
                item = self.item(row, col)
                if item:
                    text += item.text()
                if col < nco-1:
                    text += '\t'
            text += '\n'
        QApplication.clipboard().setText(text);

        
# Program starts here:
if __name__ == '__main__':
    app = QApplication(sys.argv)  
    app.setStyle('plastique')
    main = INDIGO_main()
    sys.exit(app.exec_())
    
    
    